/*

 Effects Script of the Sphere Community Game
 
 19 apr 2007 - inital version, Vincent van Beveren
 
 */
Game.log.info('Initializing Effects...');

/**
 * Effect is an object that can apply one or more modifiers to one or more targets.
 *
 * parameters
 *	modifiers		Either one modifier or an array of multiple.
 *
 * Example:
 * potionEffect = new Effect(new BasicStatModifier('hp', +10));
 */
function Effect(modifiers) {
	if (!modifiers) {
		modifiers = new Array();
	} else if (!modifiers.length) {
		modifiers = [modifiers];
	}
	this.modifiers = modifiers;
	
	/**
	 * applies one or more modifiers from source to targets.
	 *
	 * parameters
	 *  source		The source charactert
	 *  targets		The target characters, either an array of characters, or one.
	 */
	this.applyTo = function(source, targets) {
		// if a single object is passed, put it in an array
		if (!targets.length) {
			targets = [targets];
		}
		for (i = 0; i < this.modifiers.length; i++) {
			this.modifiers[i].applyTo(source, targets);
		}
	}
	
	/**
	 * Returns a nice string specifying the effects this effect has.
	 */
	this.toString = function() {
		s = "";
		for (i = 0; i < this.modifiers.length; i++) {
			s += this.modifiers[i];
			if (i < this.modifiers.lenght - 1) {
				s += ",";
			}
		}
		return s;
	}
}

/**
 * Factory for creating a simple stat modifier.
 *
 * Example, restores 20 mp:
 *
 * restoreMp = Effect.createBasicEffect('mp', +20);
 */
Effect.createBasicEffect = function(stat, change) {
	return new Effect(new BasicStatModifier(stat, change));	
}

/**
 * A modifier that changes one stat. If the stat is applied to
 * multiple characters the effect is divided by the number of
 * targets.
 *
 * parameters:
 *  stat		the name of the stat to modify, must be the
 *					same as the field name of the stat object.
 *	change	The change of the stat.
 */
function BasicStatModifier(stat, change) {
	this.stat = stat;
	this.change = change;
	
	/**
	 * Main application method. Applies the stat modifier to one or more
	 * targets. This method should never be directly invoked. 
	 *
	 * parameters:
	 *  source		The source character
	 *  targets		The targets, always an array even when only one character.
	 */
	this.applyTo = function(source, targets) {
		for (i = 0; i < targets.length; i++) {
			targets[i].stats[stat].inc(Math.ceil(this.change / targets.length));
		}
	}
	
	/**
	 * Nicely formatted change. 
	 */
	this.toString = function() {
		return this.stat + " " + (this.change > 0 ? "+" : "") + this.change;
	}
}